import datetime
import logging
from flask import Flask, render_template, request, redirect, url_for, flash, jsonify
from flask_sqlalchemy import SQLAlchemy
from flask_login import LoginManager, login_user, logout_user, login_required, current_user, UserMixin
from werkzeug.security import generate_password_hash, check_password_hash
from config import Config
from utils import json_response, admin_required, student_required

app = Flask(__name__)
app.config.from_object(Config)

# 初始化数据库
db = SQLAlchemy(app)

# 初始化登录管理
login_manager = LoginManager(app)
login_manager.login_view = 'index'

# 日志配置
logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

# 数据库模型
class User(UserMixin, db.Model):
    __tablename__ = 'users'
    id = db.Column(db.Integer, primary_key=True)
    username = db.Column(db.String(50), unique=True, nullable=False)
    password_hash = db.Column(db.String(255), nullable=False)
    role = db.Column(db.Enum('student', 'admin'), nullable=False)
    email = db.Column(db.String(100))
    full_name = db.Column(db.String(100))
    created_at = db.Column(db.DateTime, nullable=False, default=datetime.datetime.utcnow)

    def set_password(self, password):
        self.password_hash = generate_password_hash(password)

    def check_password(self, password):
        return check_password_hash(self.password_hash, password)

    def get_id(self):
        return str(self.id)


class Course(db.Model):
    __tablename__ = 'courses'
    id = db.Column(db.Integer, primary_key=True)
    course_code = db.Column(db.String(20), unique=True, nullable=False)
    name = db.Column(db.String(100), nullable=False)
    description = db.Column(db.Text)
    credits = db.Column(db.Integer, nullable=False)
    teacher = db.Column(db.String(100))
    capacity = db.Column(db.Integer, nullable=False)


class Enrollment(db.Model):
    __tablename__ = 'enrollments'
    id = db.Column(db.Integer, primary_key=True)
    student_id = db.Column(db.Integer, db.ForeignKey('users.id'), nullable=False)
    course_id = db.Column(db.Integer, db.ForeignKey('courses.id'), nullable=False)
    enroll_date = db.Column(db.DateTime, nullable=False, default=datetime.datetime.utcnow)
    status = db.Column(db.Enum('enrolled', 'dropped'), nullable=False)

    student = db.relationship('User', backref='enrollments')
    course = db.relationship('Course', backref='enrollments')


class Grade(db.Model):
    __tablename__ = 'grades'
    id = db.Column(db.Integer, primary_key=True)
    enrollment_id = db.Column(db.Integer, db.ForeignKey('enrollments.id'), nullable=False)
    grade = db.Column(db.String(5))
    updated_at = db.Column(db.DateTime)

    enrollment = db.relationship('Enrollment', backref='grade')


@login_manager.user_loader
def load_user(user_id):
    return User.query.get(int(user_id))


# 首页
@app.route('/')
def index():
    return render_template('index.html')


# 注册页
@app.route('/register', methods=['GET', 'POST'])
def register():
    if request.method == 'POST':
        data = request.get_json() or request.form
        username = data.get('username')
        password = data.get('password')
        email = data.get('email')
        full_name = data.get('full_name')
        role = data.get('role')

        if not username or not password or not role:
            return json_response(False, '用户名、密码和角色为必填项')
        if role not in ['student', 'admin']:
            return json_response(False, '角色必须是 student 或 admin')

        if User.query.filter_by(username=username).first():
            return json_response(False, '用户名已存在')

        user = User(username=username, role=role, email=email, full_name=full_name, created_at=datetime.datetime.utcnow())
        user.set_password(password)
        db.session.add(user)
        db.session.commit()
        logger.info(f'新用户注册: {username} ({role})')
        return json_response(True, '注册成功')

    return render_template('register.html')


# 登录接口
@app.route('/api/login', methods=['POST'])
def api_login():
    data = request.get_json()
    if not data:
        return json_response(False, '请求数据格式错误')

    username = data.get('username')
    password = data.get('password')

    if not username or not password:
        return json_response(False, '用户名和密码不能为空')

    user = User.query.filter_by(username=username).first()
    if user and user.check_password(password):
        login_user(user)
        logger.info(f'用户登录成功: {username}')
        return json_response(True, '登录成功', {'user_id': user.id, 'role': user.role})
    else:
        logger.warning(f'用户登录失败: {username}')
        return json_response(False, '用户名或密码错误')


# 登出
@app.route('/logout')
@login_required
def logout():
    logout_user()
    flash('您已成功登出')
    return redirect(url_for('index'))


# 用户注册接口
@app.route('/api/register', methods=['POST'])
def api_register():
    data = request.get_json()
    if not data:
        return json_response(False, '请求数据格式错误')

    username = data.get('username')
    password = data.get('password')
    email = data.get('email')
    full_name = data.get('full_name')
    role = data.get('role')

    if not username or not password or not role:
        return json_response(False, '用户名、密码和角色为必填项')
    if role not in ['student', 'admin']:
        return json_response(False, '角色必须是 student 或 admin')

    if User.query.filter_by(username=username).first():
        return json_response(False, '用户名已存在')

    user = User(username=username, role=role, email=email, full_name=full_name, created_at=datetime.datetime.utcnow())
    user.set_password(password)
    db.session.add(user)
    db.session.commit()
    logger.info(f'新用户注册: {username} ({role})')
    return json_response(True, '注册成功')


# 获取课程列表，支持筛选
@app.route('/api/courses', methods=['GET'])
@login_required
def get_courses():
    name = request.args.get('name', '', type=str)
    teacher = request.args.get('teacher', '', type=str)
    credits = request.args.get('credits', type=int)

    query = Course.query
    if name:
        query = query.filter(Course.name.like(f'%{name}%'))
    if teacher:
        query = query.filter(Course.teacher.like(f'%{teacher}%'))
    if credits is not None:
        query = query.filter(Course.credits == credits)

    courses = query.all()
    course_list = []
    for c in courses:
        enrolled_count = Enrollment.query.filter_by(course_id=c.id, status='enrolled').count()
        course_list.append({
            'id': c.id,
            'course_code': c.course_code,
            'name': c.name,
            'description': c.description,
            'credits': c.credits,
            'teacher': c.teacher,
            'capacity': c.capacity,
            'enrolled': enrolled_count
        })
    return jsonify(course_list)


# 管理员创建新课程
@app.route('/api/courses', methods=['POST'])
@login_required
@admin_required
def create_course():
    data = request.get_json()
    if not data:
        return json_response(False, '请求数据格式错误')

    course_code = data.get('course_code')
    name = data.get('name')
    description = data.get('description')
    credits = data.get('credits')
    teacher = data.get('teacher')
    capacity = data.get('capacity')

    if not course_code or not name or credits is None or capacity is None:
        return json_response(False, '课程代码、名称、学分和容量为必填项')

    if Course.query.filter_by(course_code=course_code).first():
        return json_response(False, '课程代码已存在')

    course = Course(course_code=course_code, name=name, description=description, credits=credits, teacher=teacher, capacity=capacity)
    db.session.add(course)
    db.session.commit()
    logger.info(f'管理员 {current_user.username} 创建课程 {course_code}')
    return json_response(True, '课程创建成功')


# 管理员更新课程信息
@app.route('/api/courses/<int:course_id>', methods=['PUT'])
@login_required
@admin_required
def update_course(course_id):
    course = Course.query.get_or_404(course_id)
    data = request.get_json()
    if not data:
        return json_response(False, '请求数据格式错误')

    name = data.get('name')
    description = data.get('description')
    credits = data.get('credits')
    teacher = data.get('teacher')
    capacity = data.get('capacity')

    if name:
        course.name = name
    if description is not None:
        course.description = description
    if credits is not None:
        course.credits = credits
    if teacher is not None:
        course.teacher = teacher
    if capacity is not None:
        course.capacity = capacity

    db.session.commit()
    logger.info(f'管理员 {current_user.username} 更新课程 {course.course_code}')
    return json_response(True, '课程更新成功')


# 管理员删除课程
@app.route('/api/courses/<int:course_id>', methods=['DELETE'])
@login_required
@admin_required
def delete_course(course_id):
    course = Course.query.get_or_404(course_id)
    db.session.delete(course)
    db.session.commit()
    logger.info(f'管理员 {current_user.username} 删除课程 {course.course_code}')
    return json_response(True, '课程删除成功')


# 学生选课
@app.route('/api/enrollments', methods=['POST'])
@login_required
@student_required
def enroll_course():
    data = request.get_json()
    if not data:
        return json_response(False, '请求数据格式错误')

    student_id = current_user.id
    course_id = data.get('course_id')

    if not course_id:
        return json_response(False, '课程ID不能为空')

    course = Course.query.get(course_id)
    if not course:
        return json_response(False, '课程不存在')

    # 检查是否已选
    existing = Enrollment.query.filter_by(student_id=student_id, course_id=course_id, status='enrolled').first()
    if existing:
        return json_response(False, '您已选过该课程')

    # 检查容量
    enrolled_count = Enrollment.query.filter_by(course_id=course_id, status='enrolled').count()
    if enrolled_count >= course.capacity:
        return json_response(False, '课程容量已满')

    enrollment = Enrollment(student_id=student_id, course_id=course_id, enroll_date=datetime.datetime.utcnow(), status='enrolled')
    db.session.add(enrollment)
    db.session.commit()
    logger.info(f'学生 {current_user.username} 选课 {course.course_code}')
    return json_response(True, '选课成功')


# 学生退课
@app.route('/api/enrollments/<int:enrollment_id>', methods=['DELETE'])
@login_required
@student_required
def drop_course(enrollment_id):
    enrollment = Enrollment.query.get_or_404(enrollment_id)
    if enrollment.student_id != current_user.id:
        return json_response(False, '无权操作该选课记录'), 403

    if enrollment.status == 'dropped':
        return json_response(False, '该课程已退选')

    enrollment.status = 'dropped'
    db.session.commit()
    logger.info(f'学生 {current_user.username} 退课 {enrollment.course.course_code}')
    return json_response(True, '退课成功')


# 获取当前用户的选课列表
@app.route('/api/enrollments_list', methods=['GET'])
@login_required
def enrollments_list():
    enrollments = Enrollment.query.filter_by(student_id=current_user.id).all()
    result = []
    for e in enrollments:
        result.append({
            'id': e.id,
            'course': {
                'course_code': e.course.course_code,
                'name': e.course.name,
                'teacher': e.course.teacher
            },
            'enroll_date': e.enroll_date.isoformat(),
            'status': e.status
        })
    return jsonify(result)


# 查询学生成绩
@app.route('/api/grades/<int:student_id>', methods=['GET'])
@login_required
def get_grades(student_id):
    # 只有本人或管理员可查询
    if current_user.id != student_id and current_user.role != 'admin':
        return json_response(False, '无权查看该学生成绩'), 403

    enrollments = Enrollment.query.filter_by(student_id=student_id, status='enrolled').all()
    result = []
    for e in enrollments:
        grade_obj = Grade.query.filter_by(enrollment_id=e.id).first()
        grade = grade_obj.grade if grade_obj else None
        result.append({
            'course_id': e.course.id,
            'course_name': e.course.name,
            'grade': grade
        })
    return jsonify(result)


# 管理员录入或修改成绩
@app.route('/api/grades/<int:enrollment_id>', methods=['PUT'])
@login_required
@admin_required
def update_grade(enrollment_id):
    enrollment = Enrollment.query.get_or_404(enrollment_id)
    data = request.get_json()
    if not data:
        return json_response(False, '请求数据格式错误')

    grade_value = data.get('grade')
    if grade_value is None:
        return json_response(False, '成绩不能为空')

    grade_obj = Grade.query.filter_by(enrollment_id=enrollment_id).first()
    if grade_obj:
        grade_obj.grade = grade_value
        grade_obj.updated_at = datetime.datetime.utcnow()
    else:
        grade_obj = Grade(enrollment_id=enrollment_id, grade=grade_value, updated_at=datetime.datetime.utcnow())
        db.session.add(grade_obj)

    db.session.commit()
    logger.info(f'管理员 {current_user.username} 更新成绩 enrollment_id={enrollment_id} grade={grade_value}')
    return json_response(True, '成绩更新成功')


# 获取用户信息
@app.route('/api/users/<int:user_id>', methods=['GET'])
@login_required
def get_user_info(user_id):
    if current_user.id != user_id and current_user.role != 'admin':
        return json_response(False, '无权查看该用户信息'), 403

    user = User.query.get_or_404(user_id)
    user_info = {
        'id': user.id,
        'username': user.username,
        'role': user.role,
        'email': user.email,
        'full_name': user.full_name,
        'created_at': user.created_at.strftime('%Y-%m-%d %H:%M:%S')
    }
    return jsonify(user_info)


# 用户修改个人信息
@app.route('/api/users/<int:user_id>', methods=['PUT'])
@login_required
def update_user_info(user_id):
    if current_user.id != user_id:
        return json_response(False, '无权修改该用户信息'), 403

    user = User.query.get_or_404(user_id)
    data = request.get_json()
    if not data:
        return json_response(False, '请求数据格式错误')

    email = data.get('email')
    full_name = data.get('full_name')
    password = data.get('password')

    if email is not None:
        user.email = email
    if full_name is not None:
        user.full_name = full_name
    if password:
        user.set_password(password)

    db.session.commit()
    logger.info(f'用户 {user.username} 更新个人信息')
    return json_response(True, '信息更新成功')


# 前端页面路由
@app.route('/courses')
@login_required
def courses_page():
    return render_template('courses.html')


@app.route('/my-courses')
@login_required
def my_courses_page():
    return render_template('my_courses.html')


@app.route('/grades')
@login_required
def grades_page():
    return render_template('grades.html')


@app.route('/admin/courses')
@login_required
@admin_required
def admin_courses_page():
    return render_template('admin_courses.html')


@app.route('/profile')
@login_required
def profile_page():
    return render_template('profile.html')


if __name__ == '__main__':
    # 创建数据库表
    with app.app_context():
        db.create_all()
    app.run(debug=True)
